﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using UnityEngine;
using UnityEngine.EventSystems;

//PageManager is a static class for managing page navigation and navigation history
public class PageManager : MonoBehaviour
{
    private static Stack<PageDefinition> _PageStack = new Stack<PageDefinition>();
    public static PageDefinition CurrentPage { get; private set; }

    private static bool _IsNavigating = false;

    /// <summary>
    /// Navigates forward to a new page, returing once the navigation has completed
    /// </summary>
    /// <param name="newPage">The definition of the page to navigate forward to</param>
    /// <returns></returns>
    public static async Task NavigateForward(PageDefinition newPage)
    {
        if (!_IsNavigating)
        {
            //Begin navigating
            _IsNavigating = true;
            if (CurrentPage != null)
            {
                //Store the last selected control, disable and transition from the old page, then set it inactive and push it onto the stack
                CurrentPage.PageGameObject.Disable();
                CurrentPage.PreviousFocusedGameObject = EventSystem.current.currentSelectedGameObject;
                await CurrentPage.TransFrom();
                CurrentPage.PageGameObject.SetActive(false);
                _PageStack.Push(CurrentPage);
            }

            //Set the current page to be the new one, set it to be active, transition to it and then enable it
            CurrentPage = newPage;
            CurrentPage.PageGameObject.gameObject.transform.localScale = new Vector3(0.0f, 0.0f, 1.0f);
            CurrentPage.PageGameObject.SetActive(true);
            await CurrentPage.TransTo();
            CurrentPage.PageGameObject.Enable();
            EventSystem.current.SetSelectedGameObject(null);
            CurrentPage.ReFocus();

            _IsNavigating = false;  //Navigation is finished now
        }
    }

    /// <summary>
    /// Navigates backward to the page present in the page stack, if applicable and returns once navigation has completed
    /// </summary>
    /// <returns></returns>
    public static async Task NavigateBack()
    {
        if (!_IsNavigating)
        {
            if (_PageStack.Count > 0 && CurrentPage != null && CurrentPage.PageGameObject != null)
            {
                //We've got pages we can navigate back to, so transition back from the current page and set in inactive when done
                _IsNavigating = true;
                CurrentPage.PageGameObject.Disable();
                await CurrentPage.TransBackFrom();
                CurrentPage.PageGameObject.SetActive(false);

                //Now, get the last page from the stack to be current, set it active, transition to then enable and refocus upon it
                CurrentPage = _PageStack.Pop();
                CurrentPage.PageGameObject.SetActive(true);
                await CurrentPage.TransBackTo();
                CurrentPage.PageGameObject.Enable();
                EventSystem.current.SetSelectedGameObject(null);
                CurrentPage.ReFocus();
                _IsNavigating = false;
            }
        }
    }

    /// <summary>
    /// Clears the internal page stack to reset back navigation
    /// </summary>
    public static void ClearNavigationHistory()
    {
        _PageStack.Clear();
    }
}

